`timescale 1ns / 1ps
`ifndef HEADER
`include "header.v"
`endif


///////////////////////////////////////////////////////////////
//                                                           //
// Module for inferring a 8k by 2 bit Dual Port Memory       //
//                                                           //
///////////////////////////////////////////////////////////////
 
module dpm_8kX2 (clk, we, a, dpra, di, spo, dpo); 
 
 input        clk; 
 input        we; 
 input  [12:0] a; 
 input  [12:0] dpra; 
 input  [1:0] di; 
 output [1:0] spo; 
 output [1:0] dpo; 
 
 reg [1:0] ram [8191:0]; 
 reg [1:0] spo; 
 reg [1:0] dpo; 

 always @(posedge clk)
 begin 
    if (we) 
       ram[a] <= di;
    else 
       spo <= ram[a];
    dpo <= ram[dpra];
 end  
endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Module for inferring a 16k by 1 bit Dual Port Memory      //
//                                                           //
///////////////////////////////////////////////////////////////
 
module dpm_16kX1 (clk, we, a, dpra, di, spo, dpo); 
 
 input        clk; 
 input        we; 
 input  [13:0] a; 
 input  [13:0] dpra; 
 input  di; 
 output spo; 
 output dpo; 
 
 reg ram [16383:0]; 
 reg spo; 
 reg dpo; 

 always @(posedge clk)
 begin 
    if (we) 
       ram[a] <= di;
    else 
       spo <= ram[a];
    dpo <= ram[dpra];
 end
endmodule


///////////////////////////////////////////////////////////////
//                                                           //
// Module for Dual Port Memory for 8k deep by 16 bit FIFO    //
//                                                           //
///////////////////////////////////////////////////////////////

module dpm_8kX16(CLK,WE,WADDR,RADDR,DIN,DOUT);

input CLK;
input WE;
input [12:0] WADDR;
input [12:0] RADDR;
input [15:0] DIN;
output [15:0] DOUT;

wire [15:0] DOUT;
wire [1:0] DMY0,DMY1,DMY2,DMY3,DMY4,DMY5,DMY6,DMY7;

// 8k 16 bit DPM  made from 8 2-bit 8k DPM's
dpm_8kX2 dpm_A (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[1:0]),.spo(DMY0),.dpo(DOUT[1:0])); 
dpm_8kX2 dpm_B (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[3:2]),.spo(DMY1),.dpo(DOUT[3:2])); 
dpm_8kX2 dpm_C (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[5:4]),.spo(DMY2),.dpo(DOUT[5:4])); 
dpm_8kX2 dpm_D (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[7:6]),.spo(DMY3),.dpo(DOUT[7:6])); 
dpm_8kX2 dpm_E (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[9:8]),.spo(DMY4),.dpo(DOUT[9:8])); 
dpm_8kX2 dpm_F (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[11:10]),.spo(DMY5),.dpo(DOUT[11:10])); 
dpm_8kX2 dpm_G (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[13:12]),.spo(DMY6),.dpo(DOUT[13:12])); 
dpm_8kX2 dpm_H (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[15:14]),.spo(DMY7),.dpo(DOUT[15:14])); 
 
endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Module for Dual Port Memory for 16k deep by 16 bit FIFO   //
//                                                           //
///////////////////////////////////////////////////////////////

module dpm_16kX16(CLK,WE,WADDR,RADDR,DIN,DOUT);

input CLK;
input WE;
input [13:0] WADDR;
input [13:0] RADDR;
input [15:0] DIN;
output [15:0] DOUT;

wire [15:0] DOUT;
wire DMY0,DMY1,DMY2,DMY3,DMY4,DMY5,DMY6,DMY7;
wire DMY8,DMY9,DMY10,DMY11,DMY12,DMY13,DMY14,DMY15;

// 16k 16 bit DPM  made from 16 1-bit 16k DPM's
dpm_16kX1 dpm16_A (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[0]),.spo(DMY0),.dpo(DOUT[0])); 
dpm_16kX1 dpm16_B (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[1]),.spo(DMY1),.dpo(DOUT[1])); 
dpm_16kX1 dpm16_C (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[2]),.spo(DMY2),.dpo(DOUT[2])); 
dpm_16kX1 dpm16_D (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[3]),.spo(DMY3),.dpo(DOUT[3])); 
dpm_16kX1 dpm16_E (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[4]),.spo(DMY4),.dpo(DOUT[4])); 
dpm_16kX1 dpm16_F (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[5]),.spo(DMY5),.dpo(DOUT[5])); 
dpm_16kX1 dpm16_G (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[6]),.spo(DMY6),.dpo(DOUT[6])); 
dpm_16kX1 dpm16_H (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[7]),.spo(DMY7),.dpo(DOUT[7])); 
dpm_16kX1 dpm16_I (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[8]),.spo(DMY8),.dpo(DOUT[8])); 
dpm_16kX1 dpm16_J (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[9]),.spo(DMY9),.dpo(DOUT[9])); 
dpm_16kX1 dpm16_K (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[10]),.spo(DMY10),.dpo(DOUT[10])); 
dpm_16kX1 dpm16_L (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[11]),.spo(DMY11),.dpo(DOUT[11])); 
dpm_16kX1 dpm16_M (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[12]),.spo(DMY12),.dpo(DOUT[12])); 
dpm_16kX1 dpm16_N (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[13]),.spo(DMY13),.dpo(DOUT[13])); 
dpm_16kX1 dpm16_O (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[14]),.spo(DMY14),.dpo(DOUT[14])); 
dpm_16kX1 dpm16_P (.clk(CLK),.we(WE),.a(WADDR),.dpra(RADDR),.di(DIN[15]),.spo(DMY15),.dpo(DOUT[15])); 
 
endmodule


////////////////////////////////////////////////////////////////////////
//                                                                    //
// Module for inferring a dp deep by n bit wide Dual Port Memory      //
//                                                                    //
// Values for 18 Kb blocks are:                                       //
//     Depth   Power of 2   Width                                     //
//     ---------------------------                                    //
//       512        9         36                                      //
//        1k       10         18                                      //
//        2k       11          9                                      //
//        4k       12          4                                      //
//        8k       13          2                                      //
//       16k       14          1                                      //
//                                                                    //
////////////////////////////////////////////////////////////////////////
 
module blk_dpm_dpXn (clk, we, a, dpra, di, spo, dpo); 
parameter p2depth = 9; // Default depth in words
parameter width = 36; // Default width in bits
 
 input        clk; 
 input        we; 
 input  [p2depth-1:0] a; 
 input  [p2depth-1:0] dpra; 
 input  [width-1:0] di; 
 output [width-1:0] spo; 
 output [width-1:0] dpo; 
 
 reg    [width-1:0] ram [(1<<p2depth)-1:0]; 
 reg [width-1:0] spo; 
 reg [width-1:0] dpo; 

 always @(posedge clk)
 begin 
    if (we) 
       ram[a] <= di;
    else 
       spo <= ram[a];
    dpo <= ram[dpra];
 end  
endmodule

module dpm_1kx16(clk,we,a,dpra,di,dpo);
input  clk; 
input  we; 
input  [9:0] a; 
input  [9:0] dpra; 
input  [15:0] di; 
output [15:0] dpo; 

wire [15:0] dpo; 
wire [15:0] spo; 

defparam mem_1kx16.p2depth = 10;
defparam mem_1kx16.width = 16;
blk_dpm_dpXn mem_1kx16(.clk(clk),.we(we),.a(a),.dpra(dpra),.di(di),.spo(spo),.dpo(dpo));
endmodule



///////////////////////////////////////////////////////////////
//                                                           //
// Module for inferring a                                    //
// dp deep by n bit wide Dual Port Memory                    //
// Using distributed RAM                                     //
//                                                           //
///////////////////////////////////////////////////////////////

// synthesis attribute ram_style of dst_dpm_dpXn is distributed
// synthesis attribute ram_style of dst_dpm_dpXn_1 is distributed
// synthesis attribute ram_style of dst_dpm_dpXn_2 is distributed
// synthesis attribute ram_style of dst_dpm_dpXn_3 is distributed
// synthesis attribute ram_style of dst_dpm_dpXn_4 is distributed
// synthesis attribute ram_style of dst_dpm_dpXn_5 is distributed
// synthesis attribute ram_style of dst_dpm_dpXn_6 is distributed
// synthesis attribute ram_style of dst_dpm_dpXn_7 is distributed
// synthesis attribute ram_style of dst_dpm_dpXn_8 is distributed
// synthesis attribute ram_style of dst_dpm_dpXn_9 is distributed
 
module dst_dpm_dpXn (clk, we, a, dpra, di, spo, dpo); 
parameter p2depth = 6; // Default depth in words
parameter width = 16; // Default width in bits
 
 input        clk; 
 input        we; 
 input  [p2depth-1:0] a; 
 input  [p2depth-1:0] dpra; 
 input  [width-1:0] di; 
 output [width-1:0] spo; 
 output [width-1:0] dpo; 
 
 reg    [width-1:0] ram [(1<<p2depth)-1:0]; 
 reg [width-1:0] spo; 
 reg [width-1:0] dpo; 

 always @(posedge clk)
 begin 
    if (we) 
       ram[a] <= di;
    else 
       spo <= ram[a];
    dpo <= ram[dpra];
 end  
endmodule


///////////////////////////////////////////////////////////////
//                                                           //
// Module for a distributed RAM FIFO                         //
//                                                           //
///////////////////////////////////////////////////////////////

module fifo_dpXn_dst(clk,rst,push,pop,mt,full,amt,af,wterr,rderr,din,dout,cnt);
parameter p2depth = 6; // Default depth in words expressed as a power of 2 (ie 2^6 = 64)
parameter width = 16; // Default width in bits
parameter almostmt = 4; // 
parameter almostfull = (1<<p2depth) - 4; // 

input clk;
input rst;
input push;
input pop;
input [width-1:0] din;
output mt;
output full;
output amt;
output af;
output wterr;
output rderr;
output [width-1:0] dout;
output [p2depth-1:0] cnt;

wire [width-1:0] spo;
wire [p2depth-1:0] waddr;
wire [p2depth-1:0] raddr;
wire ce;
wire dir;
wire ppop;
wire ppush;
wire wterr;
wire rderr;

assign   mt = ~|cnt;
assign   full = &cnt;
assign   amt = (cnt <= almostmt);
assign   af = (cnt >= almostfull);
//assign   mt = (cnt == 6'h00);
//assign   full = (cnt == 6'h3F);
//assign   amt = (cnt <= 6'h4);
//assign   af = (cnt >= 6'h3C);

defparam dp_fifo.p2depth = p2depth;
defparam dp_fifo.width = width;
defparam ffdp_wrt_ptr.width = p2depth;
defparam ffdp_rd_ptr.width = p2depth;
defparam ffdp_cntr.width = p2depth;

cnt_up ffdp_wrt_ptr (.clk(clk),.inc(ppush),.r(rst),.count(waddr));
// dp X n Dual Port RAM using distributed memory for fifo
dst_dpm_dpXn dp_fifo(.clk(clk),.we(ppush),.a(waddr),.dpra(raddr),.di(din),.spo(spo),.dpo(dout)); 
cnt_up  ffdp_rd_ptr (.clk(clk),.inc(ppop),.r(rst),.count(raddr));

cnt_up_dwn ffdp_cntr (.clk(clk),.ce(ce),.dir(dir),.r(rst),.count(cnt));

ffpp_ctrl ffdp_ctrl (.mt(mt),.full(full),.push(push),.pop(pop),.ce(ce),.dir(dir),.ppop(ppop),.ppush(ppush),.wterr(wterr),.rderr(rderr));

endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Module for controlling FIFO pushes pops and counter       //
//                                                           //
///////////////////////////////////////////////////////////////

module ffpp_ctrl(mt,full,push,pop,ce,dir,ppop,ppush,wterr,rderr);

input mt;
input full;
input push;
input pop;
output ce;
output dir;
output ppop;
output ppush;
output wterr;
output rderr;

reg ce;
reg dir;
reg ppop;
reg ppush;
reg wterr;
reg rderr;

always @ (mt or full or push or pop)
begin
  casex ({mt,full,push,pop})
     4'b01x1,4'b0001:
       begin                 // count down
         ce <= 1'b1;
         dir <= 1'b0;
         ppop <= 1'b1;         // Increment read address
         ppush <= 1'b0;         // Don't increment write address
       end
     4'b101x,4'b0010:
       begin                 // count up
         ce <= 1'b1;         
         dir <= 1'b1;
         ppop <= 1'b0;         // Don't increment read address
         ppush <= 1'b1;         // Increment write address
       end
     4'b0011:
       begin                 // hold
         ce <= 1'b0;
         dir <= 1'b1;
         ppop <= 1'b1;         // Increment read address
         ppush <= 1'b1;         // Increment write address
       end
     default:                 // hold
       begin
         ce <= 1'b0;
         dir <= 1'b1;
         ppop <= 1'b0;         // Don't increment address
         ppush <= 1'b0;         // Don't increment write address
       end
  endcase
end
always @ (mt or full or push or pop)
begin
  casex ({mt,full,push,pop})
     4'b011x:
       begin                 // write error
         wterr <= 1'b1;
         rderr <= 1'b0;
       end
     4'b10x1:
       begin                 // read error
         wterr <= 1'b0;
         rderr <= 1'b1;
       end
     4'b11xx:
       begin                 // mt and full error
         wterr <= 1'b1;
         rderr <= 1'b1;
       end
     default:                 // no error
       begin
         wterr <= 1'b0;
         rderr <= 1'b0;
       end
  endcase
end
endmodule

/////////////////////////////////////////////////////////////////
////                                                           //
//// Module for a distributed RAM FIFO with UnPoP Feature      //
////                                                           //
/////////////////////////////////////////////////////////////////
//
//module fifoup_dpXn_dst(clk,rst,push,pop,unpop,mt,full,amt,af,wterr,rderr,din,dout,cnt);
//parameter p2depth = 6; // Default depth in words expressed as a power of 2 (ie 2^6 = 64)
//parameter width = 16; // Default width in bits
//parameter almostmt = 4; // 
//parameter almostfull = (1<<p2depth) - 4; // 
//
//input clk;
//input rst;
//input push;
//input pop;
//input unpop;
//input [width-1:0] din;
//output mt;
//output full;
//output amt;
//output af;
//output wterr;
//output rderr;
//output [width-1:0] dout;
//output [p2depth-1:0] cnt;
//
//wire [width-1:0] spo;
//wire [p2depth-1:0] waddr;
//wire [p2depth-1:0] raddr;
//wire ce;
//wire dir;
//wire ppop;
//wire punpop;
//wire ppush;
//wire wterr;
//wire rderr;
//wire willbemt;
//
//assign   cnt = waddr - raddr;
//assign   mt = ~|cnt;
//assign   willbemt = ~|(cnt+2);
//assign   full = &cnt;
//assign   amt = (cnt <= almostmt);
//assign   af = (cnt >= almostfull);
////assign   mt = (cnt == 6'h00);
////assign   full = (cnt == 6'h3F);
////assign   amt = (cnt <= 6'h4);
////assign   af = (cnt >= 6'h3C);
//
//defparam dp_fifo.p2depth = p2depth;
//defparam dp_fifo.width = width;
//defparam ffdp_wrt_ptr.width = p2depth;
//defparam ffdp_rd_ptr.width = p2depth;
//defparam ffdp_cntr.width = p2depth;
//
//cnt_up ffdp_wrt_ptr (.clk(clk),.inc(ppush),.r(rst),.count(waddr));
//// dp X n Dual Port RAM using distributed memory for fifo
//dst_dpm_dpXn dp_fifo(.clk(clk),.we(ppush),.a(waddr),.dpra(raddr),.di(din),.spo(spo),.dpo(dout)); 
//inc_dec  ffdp_rd_ptr (.clk(clk),.inc(ppop),.dec(punpop),.r(rst),.count(raddr));
//
//ffpp_ctrl ffdp_ctrl (.mt(mt),.willbemt(willbemt),.full(full),.push(push),.pop(pop),unpop(unpop),
//                     .ppop(ppop),.punpop(punpop),.ppush(ppush),.wterr(wterr),.rderr(rderr));
//
//endmodule
//
/////////////////////////////////////////////////////////////////
////                                                           //
//// Module for controlling FIFO pushes pops and counter       //
////                                                           //
/////////////////////////////////////////////////////////////////
//
//module ffppu_ctrl(mt,willbemt,full,push,pop,unpop,ppop,punpop,ppush,wterr,rderr);
//
//input mt;
//input willbemt;
//input full;
//input push;
//input pop;
//input unpop;
//output ppop;
//output punpop;
//output ppush;
//output wterr;
//output rderr;
//
//reg ppop;
//reg punpop;
//reg ppush;
//reg wterr;
//reg rderr;
//
//always @ (full or push or unpop or willbemt)
//begin
//   casex ({full,push,unpop,willbemt})
//      4'b010x,5'b0110:
//         ppush <= 1'b1;         // Increment write address
//      default:
//         ppush <= 1'b0;         // Don't increment write address
//   endcase
//end
//always @ (mt or full or pop or unpop)
//begin
//  casex ({mt,full,pop,unpop})
//      4'b0x10:
//         begin
//            ppop <= 1'b1;         // Increment read address
//            punpop <= 1'b0;       // Don't decrement read address
//         end
//      4'bx001:
//         begin
//            ppop <= 1'b0;         // Don't increment read address
//            punpop <= 1'b1;       // Decrement read address
//         end
//      default:                 // hold
//         begin
//            ppop <= 1'b0;         // Don't increment address
//            punpop <= 1'b0;       // Don't decrement read address
//         end
//   endcase
//end
//always @ (mt or full or push or pop or unpop or willbemt)
//begin
//  casex ({mt,full,push,pop,unpop,willbemt})
//     6'b011xxx,6'bxx1x11:
//       begin                 // write error
//         wterr <= 1'b1;
//         rderr <= 1'b0;
//       end
//     6'b10x1xx,6'b01x01x:
//       begin                 // read error
//         wterr <= 1'b0;
//         rderr <= 1'b1;
//       end
//     6'b11xxxx:
//       begin                 // mt and full error
//         wterr <= 1'b1;
//         rderr <= 1'b1;
//       end
//     default:                 // no error
//       begin
//         wterr <= 1'b0;
//         rderr <= 1'b0;
//       end
//  endcase
//end
//endmodule


///////////////////////////////////////////////////////////////
//                                                           //
// Module Block RAM FIFO                                     //
//                                                           //
///////////////////////////////////////////////////////////////

module fifo_dpXn_blk(clk,rst,push,pop,mt,full,amt,af,wterr,rderr,din,dout,cnt);
parameter p2depth = 9; // Default depthin words expressed as a power of 2 (ie 2^6 = 64)
parameter width = 36; // Default width in bits
parameter almostmt = 32; // 
parameter almostfull = (1<<p2depth) - 64; // 

input clk;
input rst;
input push;
input pop;
input [width-1:0] din;
output mt;
output full;
output amt;
output af;
output wterr;
output rderr;
output [width-1:0] dout;
output [p2depth-1:0] cnt;

wire [width-1:0] spo;
wire [p2depth-1:0] waddr;
wire [p2depth-1:0] raddr;
wire ce;
wire dir;
wire ppop;
wire push;
wire wterr;
wire rderr;

assign   mt = ~|cnt;
assign   full = &cnt;
assign   amt = (cnt <= almostmt);
assign   af = (cnt >= almostfull);
//assign   mt = (cnt == 9'h000);
//assign   full = (cnt == 9'h1FF);
//assign   amt = (cnt <= 9'h020);
//assign   af = (cnt >= 9'h1C0);

defparam dp_fifo.p2depth = p2depth;
defparam dp_fifo.width = width;
defparam ffdp_wrt_ptr.width = p2depth;
defparam ffdp_rd_ptr.width = p2depth;
defparam ffdp_cntr.width = p2depth;

cnt_up ffdp_wrt_ptr (.clk(clk),.inc(ppush),.r(rst),.count(waddr));
// dp X n Dual Port Block RAM for FIFO
blk_dpm_dpXn dp_fifo(.clk(clk),.we(ppush),.a(waddr),.dpra(raddr),.di(din),.spo(spo),.dpo(dout)); 
cnt_up  ffdp_rd_ptr (.clk(clk),.inc(ppop),.r(rst),.count(raddr));

cnt_up_dwn ffdp_cntr (.clk(clk),.ce(ce),.dir(dir),.r(rst),.count(cnt));

ffpp_ctrl ffdp_ctrl (.mt(mt),.full(full),.push(push),.pop(pop),.ce(ce),.dir(dir),.ppop(ppop),.ppush(ppush),.wterr(wterr),.rderr(rderr));

endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Module Block 8K x 16  FIFO                                //
//                                                           //
///////////////////////////////////////////////////////////////

module fifo_8Kx16(clk,rst,push,pop,mt,full,amt,af,wterr,rderr,din,dout,cnt);
parameter almostmt = 32; // 
parameter almostfull = 8128; // 

input clk;
input rst;
input push;
input pop;
input [15:0] din;
output mt;
output full;
output amt;
output af;
output wterr;
output rderr;
output [15:0] dout;
output [12:0] cnt;

wire [12:0] waddr;
wire [12:0] raddr;
wire ce;
wire dir;
wire ppop;
wire push;
wire wterr;
wire rderr;

assign   mt = ~|cnt;
assign   full = &cnt;
assign   amt = (cnt <= almostmt);
assign   af = (cnt >= almostfull);

defparam ff_wrt_ptr.width = 13;
defparam ff_rd_ptr.width = 13;
defparam ff_cntr.width = 13;

cnt_up ff_wrt_ptr(.clk(clk),.inc(ppush),.r(rst),.count(waddr));
// 8k X 16 Dual Port Block RAM for FIFO
dpm_8kX16 ff_8k(.CLK(clk),.WE(ppush),.WADDR(waddr),.RADDR(raddr),.DIN(din),.DOUT(dout));
cnt_up  ff_rd_ptr(.clk(clk),.inc(ppop),.r(rst),.count(raddr));

cnt_up_dwn ff_cntr(.clk(clk),.ce(ce),.dir(dir),.r(rst),.count(cnt));

ffpp_ctrl ff_ctrl(.mt(mt),.full(full),.push(push),.pop(pop),.ce(ce),.dir(dir),.ppop(ppop),.ppush(ppush),.wterr(wterr),.rderr(rderr));

endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Module Block 16K x 16  FIFO                               //
//                                                           //
///////////////////////////////////////////////////////////////

module fifo_16Kx16(clk,rst,push,pop,mt,full,amt,af,wterr,rderr,din,dout,cnt);
parameter almostmt = 64; // 
parameter almostfull = 16320; // 

input clk;
input rst;
input push;
input pop;
input [15:0] din;
output mt;
output full;
output amt;
output af;
output wterr;
output rderr;
output [15:0] dout;
output [13:0] cnt;

wire [13:0] waddr;
wire [13:0] raddr;
wire ce;
wire dir;
wire ppop;
wire push;
wire wterr;
wire rderr;

assign   mt = ~|cnt;
assign   full = &cnt;
assign   amt = (cnt <= almostmt);
assign   af = (cnt >= almostfull);

defparam ff_wrt_ptr.width = 14;
defparam ff_rd_ptr.width = 14;
defparam ff_cntr.width = 14;

cnt_up ff_wrt_ptr(.clk(clk),.inc(ppush),.r(rst),.count(waddr));
// 16k X 16 Dual Port Block RAM for FIFO
dpm_16kX16 ff_16k(.CLK(clk),.WE(ppush),.WADDR(waddr),.RADDR(raddr),.DIN(din),.DOUT(dout));
cnt_up  ff_rd_ptr(.clk(clk),.inc(ppop),.r(rst),.count(raddr));

cnt_up_dwn ff_cntr(.clk(clk),.ce(ce),.dir(dir),.r(rst),.count(cnt));

ffpp_ctrl ff_ctrl(.mt(mt),.full(full),.push(push),.pop(pop),.ce(ce),.dir(dir),.ppop(ppop),.ppush(ppush),.wterr(wterr),.rderr(rderr));

endmodule


module macsrl(clk,ce,mhin,din,mhout,dout);
///////////////////////////////////////////////////////////////
//                                                           //
// 16 bit wide 5 words deep shift register                   //
//                                                           //
///////////////////////////////////////////////////////////////
input  clk;
input  ce;
input  [1:0]mhin;
input  [15:0]din;
output [1:0]mhout;
output [15:0]dout;

wire [1:0]mhout;
wire [15:0]dout;
reg [1:0]h1,h2,m3,m2,m1;
reg [15:0]seq,com,mac3,mac2,mac1;

assign  mhout = m1;
assign  dout = mac1;
always @(posedge clk)
begin
   if(ce)
      begin
         seq <= din;   // sequential packet ID
         com <= seq;   // command word
         mac3 <= com;  // Octet 4 and 5
         mac2 <= mac3; // Octet 2 and 3
         mac1 <= mac2; // Octet 0 and 1
         h2 <= mhin;   // sequential packet ID
         h1 <= h2;   // command word
         m3 <= h1;  // Octet 4 and 5
         m2 <= m3; // Octet 2 and 3
         m1 <= m2; // Octet 0 and 1
      end
end
endmodule


/////////////////////////////////////////////////////////////////
////                                                           //
//// Incr/Decr counter with synchonous reset and parameterized width  //
////                                                           //
/////////////////////////////////////////////////////////////////
//
//module inc_dec(clk,inc,dec,r,count);
//parameter width = 16; // Default counter width in bits
//
//input clk;
//input inc;
//input dec;
//input r;
//output [width-1:0] count;
//
//reg [width-1:0] count;
//
//always @ (posedge clk)
//begin
//   if (r)
//      count <= 0;
//   else if (inc)
//      count <= count + 1;
//   else if (dec)
//      count <= count - 1;
//   else
//      count <= count;
//end
//endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Up counter with synchonous reset and parameterized width  //
//                                                           //
///////////////////////////////////////////////////////////////

module cnt_up(clk,inc,r,count);
parameter width = 16; // Default counter width in bits

input clk;
input inc;
input r;
output [width-1:0] count;

reg [width-1:0] count;

always @ (posedge clk)
begin
   if (r)
      count <= 0;
   else if (inc)
      count <= count + 1;
   else
      count <= count;
end
endmodule


//module cnt_up_8(clk,inc,r,count);
//// 8 Bit instance
//input clk,inc,r;
//output [7:0] count;
//defparam cu_8.width = 8;
//cnt_up   cu_8 (.clk(clk),.inc(inc),.r(r),.count(count));
//endmodule

//module cnt_up_9(clk,inc,r,count);
//// 9 Bit instance
//input clk,inc,r;
//output [8:0] count;
//defparam cu_9.width = 9;
//cnt_up   cu_9 (.clk(clk),.inc(inc),.r(r),.count(count));
//endmodule

module cnt_up_10(clk,inc,r,count);
// 10 Bit instance
input clk,inc,r;
output [9:0] count;
defparam cu_10.width = 10;
cnt_up   cu_10 (.clk(clk),.inc(inc),.r(r),.count(count));
endmodule

module cnt_up_13(clk,inc,r,count);
// 13 Bit instance
input clk,inc,r;
output [12:0] count;
defparam cu_13.width = 13;
cnt_up   cu_13 (.clk(clk),.inc(inc),.r(r),.count(count));
endmodule

//module cnt_up_32(clk,inc,r,count);
//// 32 Bit instance
//input clk,inc,r;
//output [31:0] count;
//defparam cu_32.width = 32;
//cnt_up   cu_32 (.clk(clk),.inc(inc),.r(r),.count(count));
//endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Loadable up counter with synchonous reset and             //
// parameterized width                                       //
//                                                           //
///////////////////////////////////////////////////////////////

module cnt_up_l(clk,inc,r,l,din,count);
parameter width = 16; // Default counter width in bits

input clk;
input inc;
input r;
input l;
input [width-1:0] din;
output [width-1:0] count;

reg [width-1:0] count;

always @ (posedge clk)
begin
   if (r)
      count <= 0;
   else if (l)
      count <= din;
   else if (inc)
      count <= count + 1;
   else
      count <= count;
end
endmodule

module cnt_up_l_10(clk,inc,r,l,din,count);
// 10 Bit instance
input clk,inc,r,l;
input [9:0] din;
output [9:0] count;
defparam cul_10.width = 10;
cnt_up_l cul_10 (.clk(clk),.inc(inc),.r(r),.l(l),.din(din),.count(count));
endmodule

module cnt_up_l_13(clk,inc,r,l,din,count);
// 13 Bit instance
input clk,inc,r,l;
input [12:0] din;
output [12:0] count;
defparam cul_13.width = 13;
cnt_up_l cul_13 (.clk(clk),.inc(inc),.r(r),.l(l),.din(din),.count(count));
endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Loadable down counter with synchonous reset and           //
// parameterized width                                       //
//                                                           //
///////////////////////////////////////////////////////////////

module cnt_dn_l(clk,dec,r,l,din,count);
parameter width = 16; // Default counter width in bits

input clk;
input dec;
input r;
input l;
input [width-1:0] din;
output [width-1:0] count;

reg [width-1:0] count;

always @ (posedge clk)
begin
   if (r)
      count <= 0;
   else if (l)
      count <= din;
   else if (dec)
      count <= count - 1;
   else
      count <= count;
end
endmodule

module cnt_dn_l_13(clk,dec,r,l,din,count);
// 13 Bit instance
input clk,dec,r,l;
input [12:0] din;
output [12:0] count;
defparam cdl_13.width = 13;
cnt_dn_l cdl_13 (.clk(clk),.dec(dec),.r(r),.l(l),.din(din),.count(count));
endmodule


///////////////////////////////////////////////////////////////
//                                                           //
// Specialized counter !!!!                                  //
// Up counter with synchonous reset to 0 or 1                //
// and parameterized width                                   //
//                                                           //
///////////////////////////////////////////////////////////////

module cnt_up_r10(clk,inc,r,count);
parameter width = 16; // Default counter width in bits

input clk;
input inc;
input r;
output [width-1:0] count;

reg [width-1:0] count;

always @ (posedge clk)
begin
   if (r)
      if(inc)
         count <= 1;
      else
         count <= 0;
   else if (inc)
      count <= count + 1;
   else
      count <= count;
end
endmodule

//module cnt_up_r10_13(clk,inc,r,count);
//// 13 Bit instance
//input clk,inc,r;
//output [12:0] count;
//defparam cu_r10_13.width = 13;
//cnt_up_r10   cu_r10_13 (.clk(clk),.inc(inc),.r(r),.count(count));
//endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Bi-Directional Register with output enable                //
//                                                           //
///////////////////////////////////////////////////////////////

//module bi_dir_reg(CLK, DIR, EN, A, B);
//
//    input CLK;
//    input DIR;
//    input EN;
//    inout [15:0] A;
//    inout [15:0] B;
//   
//reg [15:0] BDREG;
//
//assign A = (EN && DIR) ? BDREG : 16'hZZZZ;
//assign B = (EN && !DIR) ? BDREG : 16'hZZZZ;
//
//always @(posedge CLK)
//begin
//   if (DIR)
//      BDREG <= B;
//   else
//      BDREG <= A;
//end      
//endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Register with output enable                               //
//                                                           //
///////////////////////////////////////////////////////////////

//module loopback(clk, ena, lp_in, lp_out);
//
//    input clk;
//    input ena;
//    input [15:0] lp_in;
//   output [15:0] lp_out;
//
//reg [15:0] dreg;
//reg [15:0] lp_out;
//
//always @(posedge clk)
//      dreg <= lp_in;
//always @(ena or dreg)
//   if (ena)      //active high enable
//      lp_out = dreg;
//   else
//      lp_out = 16'bZ;
//         
//endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Register with clock enable and synchronous reset and      //
// parameterized width                                       //
//                                                           //
///////////////////////////////////////////////////////////////

module reg_ce(clk,ce,r,din,dout);
parameter width = 16; // Default counter width in bits

input clk;
input ce;
input r;
input [width-1:0] din;
output [width-1:0] dout;

reg [width-1:0] dout;

always @ (posedge clk)
begin
   if (r)
      dout <= 0;
   else if (ce)
      dout <= din;
   else
      dout <= dout;
end
endmodule

//module reg_ce_4(clk,ce,r,din,dout);
//// 4 Bit instance
//input clk,ce,r;
//input [3:0] din;
//output [3:0] dout;
//defparam rce_4.width = 4;
//reg_ce   rce_4 (.clk(clk),.ce(ce),.r(r),.din(din),.dout(dout));
//endmodule

//module reg_ce_8(clk,ce,r,din,dout);
//// 8 Bit instance
//input clk,ce,r;
//input [7:0] din;
//output [7:0] dout;
//defparam rce_8.width = 8;
//reg_ce   rce_8 (.clk(clk),.ce(ce),.r(r),.din(din),.dout(dout));
//endmodule

module reg_ce_10(clk,ce,r,din,dout);
// 10 Bit instance
input clk,ce,r;
input [9:0] din;
output [9:0] dout;
defparam rce_10.width = 10;
reg_ce   rce_10 (.clk(clk),.ce(ce),.r(r),.din(din),.dout(dout));
endmodule

module reg_ce_13(clk,ce,r,din,dout);
// 13 Bit instance
input clk,ce,r;
input [12:0] din;
output [12:0] dout;
defparam rce_13.width = 13;
reg_ce   rce_13 (.clk(clk),.ce(ce),.r(r),.din(din),.dout(dout));
endmodule

//module reg_ce_16(clk,ce,r,din,dout);
//// 16 Bit instance
//input clk,ce,r;
//input [15:0] din;
//output [15:0] dout;
//defparam rce_16.width = 16;
//reg_ce   rce_16 (.clk(clk),.ce(ce),.r(r),.din(din),.dout(dout));
//endmodule
//
//module reg_ce_32(clk,ce,r,din,dout);
//// 32 Bit instance
//input clk,ce,r;
//input [31:0] din;
//output [31:0] dout;
//defparam rce_32.width = 32;
//reg_ce   rce_32 (.clk(clk),.ce(ce),.r(r),.din(din),.dout(dout));
//endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Up/Down counter with clock enable synchonous reset and    //
// parameterized width                                       //
//                                                           //
///////////////////////////////////////////////////////////////

module cnt_up_dwn(clk,ce,dir,r,count);
parameter width = 16; // Default counter width in bits

input clk;
input ce;
input dir;
input r;
output [width-1:0] count;

reg [width-1:0] count;

initial
begin
   count = 0;
end

always @ (posedge clk)
begin
   if (r)
      count <= 0;
   else if (ce)
      if (dir)
         count <= count + 1;
      else
         count <= count - 1;
   else
      count <= count;
end
endmodule

//module cnt_up_dwn_9(clk,ce,dir,r,count);
//// 9 Bit instance
//input clk,ce,dir,r;
//output [8:0] count;
//defparam cud_9.width = 9;
//cnt_up_dwn   cud_9 (.clk(clk),.ce(ce),.dir(dir),.r(r),.count(count));
//endmodule
//
//module cnt_up_dwn_13(clk,ce,dir,r,count);
//// 13 Bit instance
//input clk,ce,dir,r;
//output [12:0] count;
//defparam cud_13.width = 13;
//cnt_up_dwn   cud_13 (.clk(clk),.ce(ce),.dir(dir),.r(r),.count(count));
//endmodule

///////////////////////////////////////////////////////////////
// Loadable                                                  //
// Up/Down counter with clock enable synchonous reset and    //
// parameterized width                                       //
//                                                           //
///////////////////////////////////////////////////////////////

module cnt_up_dwn_l(clk,ce,dir,r,l,din,count);
parameter width = 16; // Default counter width in bits

input clk;
input ce;
input dir;
input r;
input l;
input [width-1:0] din;
output [width-1:0] count;

reg [width-1:0] count;

always @ (posedge clk)
begin
   if (r)
      count <= 0;
   else if (ce)
      if (l)
         count <= din;
      else
         if (dir)
            count <= count + 1;
         else
            count <= count - 1;
   else
      count <= count;
end
endmodule


///////////////////////////////////////////////////////////////
//                                                           //
// 2 to 1 MUX with parameterized width                       //
//                                                           //
///////////////////////////////////////////////////////////////

//module mux2to1(sel,ain,bin,out);
//parameter width = 16; // Default data width in bits
//
//input sel;
//input [width-1:0] ain,bin;
//output [width-1:0] out;
//
//reg [width-1:0] out;
//
//always @ (sel or ain or bin)
//begin
//   if (sel)
//      out <= bin;
//   else 
//      out <= ain;
//end
//endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// 2 to 1 MUX with registered output and parameterized width //
//                                                           //
///////////////////////////////////////////////////////////////

//module mux2to1_reg(clk,sel,ain,bin,out);
//parameter width = 16; // Default data width in bits
//
//input clk;
//input sel;
//input [width-1:0] ain,bin;
//output [width-1:0] out;
//
//reg [width-1:0] out;
//
//always @ (posedge clk)
//begin
//   if (sel)
//      out <= bin;
//   else 
//      out <= ain;
//end
//endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// 2 to 1 MUX with registered and tri-stated output          //
// 16 bit fixed width                                        //
//                                                           //
///////////////////////////////////////////////////////////////

//module mux2to1_reg_3st(clk,sel,ena,ain,bin,out);
//
//input clk;
//input sel;
//input ena;
//input [15:0] ain,bin;
//output [15:0] out;
//
//reg [15:0] out;
//
//always @ (posedge clk)
//begin
//   if(ena)
//      begin
//         if (sel)
//            out <= bin;
//         else 
//            out <= ain;
//      end
//   else
//      out <= 16'hzzzz;
//end
//endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// 3 to 1 MUX with parameterized width                       //
//                                                           //
///////////////////////////////////////////////////////////////

module mux3to1(sela,selb,selc,ain,bin,cin,out);
parameter width = 16; // Default data width in bits

input sela,selb,selc;
input [width-1:0] ain,bin,cin;
output [width-1:0] out;

reg [width-1:0] out;

always @ (sela or selb or selc or ain or bin or cin)
begin
   if (sela)
      out <= ain;
   else if(selb) 
      out <= bin;
   else if(selc) 
      out <= cin;
   else 
      out <= ain;
end
endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// 3 to 1 MUX with registered output and parameterized width //
//                                                           //
///////////////////////////////////////////////////////////////

module mux3to1_reg(clk,sela,selb,selc,ain,bin,cin,out);
parameter width = 16; // Default data width in bits

input clk;
input sela,selb,selc;
input [width-1:0] ain,bin,cin;
output [width-1:0] out;

reg [width-1:0] out;

always @ (posedge clk)
begin
   if (sela)
      out <= ain;
   else if(selb) 
      out <= bin;
   else if(selc) 
      out <= cin;
   else 
      out <= ain;
end
endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// 4 to 1 MUX with parameterized width                       //
//                                                           //
///////////////////////////////////////////////////////////////

//module mux4to1(sel,ain,bin,cin,din,out);
//parameter width = 16; // Default data width in bits
//
//input [1:0] sel;
//input [width-1:0] ain,bin,cin,din;
//output [width-1:0] out;
//
//reg [width-1:0] out;
//
//always @ (sel or ain or bin or cin or din)
//begin
//   case (sel)
//      2'd0:
//         out <= ain;
//      2'd1:
//         out <= bin;
//      2'd2:
//         out <= cin;
//      2'd3:
//         out <= din;
//      default:
//         out <= ain;
//   endcase
//end
//endmodule


///////////////////////////////////////////////////////////////
//                                                           //
// 8 to 1 MUX with parameterized width                       //
//                                                           //
///////////////////////////////////////////////////////////////

module mux8to1(sel,ain,bin,cin,din,ein,fin,gin,hin,out);
parameter width = 16; // Default data width in bits

input [2:0] sel;
input [width-1:0] ain,bin,cin,din,ein,fin,gin,hin;
output [width-1:0] out;

reg [width-1:0] out;

always @ (sel or ain or bin or cin or din or ein or fin or gin or hin)
begin
   case (sel)
      3'd0:
         out <= ain;
      3'd1:
         out <= bin;
      3'd2:
         out <= cin;
      3'd3:
         out <= din;
      3'd4:
         out <= ein;
      3'd5:
         out <= fin;
      3'd6:
         out <= gin;
      3'd7:
         out <= hin;
      default:
         out <= ain;
   endcase
end
endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Three Deep Rotating Shift Register                        //
// parameterized width                                       //
//                                                           //
///////////////////////////////////////////////////////////////

module shft_3_reg(c,s,l,d,q);
parameter width = 16; // Default data width in bits

input c;
input s;
input l;
input [width-1:0] d;
output [width-1:0] q;

reg [width-1:0] q;
reg [width-1:0] qa;
reg [width-1:0] qb;

always @ (posedge c)
begin
   if (s || l)
      if (l)
         begin
            qa <= d;
            qb <= qa;
            q <= qb;
         end
      else
         begin
            qa <= q;
            qb <= qa;
            q <= qb;
         end
   else
      begin
         qa <= qa;
         qb <= qb;
         q  <= q;
      end
end
endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Four Deep Rotating Shift Register                         //
// parameterized width                                       //
//                                                           //
///////////////////////////////////////////////////////////////

module shft_4_reg(c,s,l,d,q);
parameter width = 16; // Default data width in bits

input c;
input s;
input l;
input [width-1:0] d;
output [width-1:0] q;

reg [width-1:0] q;
reg [width-1:0] qa;
reg [width-1:0] qb;
reg [width-1:0] qc;

always @ (posedge c)
begin
   if (s || l)
      if (l)
         begin
            qa <= d;
            qb <= qa;
            qc <= qb;
            q  <= qc;
         end
      else
         begin
            qa <= q;
            qb <= qa;
            qc <= qb;
            q  <= qc;
         end
   else
      begin
         qa <= qa;
         qb <= qb;
         qc <= qc;
         q  <= q;
      end
end
endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Three Deep Rotating Shift Register with 3-to-1 Mux        //
// parameterized width                                       //
//                                                           //
///////////////////////////////////////////////////////////////

//module shft_3_mux3to1(c,s,l,p,a,b,q);
//parameter width = 16; // Default data width in bits
//
//input c;
//input s;
//input l;
//input p;
//input [width-1:0] a,b;
//output [width-1:0] q;
//
//reg [width-1:0] q;
//reg [width-1:0] qa;
//reg [width-1:0] qb;
//
//always @ (posedge c)
//begin
//   if (s || l)
//      casex ({s,l,p})
//         3'b010:
//            begin
//               qa <= a;
//               qb <= qa;
//               q <= qb;
//            end
//         3'b011:
//            begin
//               qa <= b;
//               qb <= qa;
//               q <= qb;
//            end
//         3'b1xx:
//            begin
//               qa <= q;
//               qb <= qa;
//               q <= qb;
//            end
//         default:
//            begin
//               qa <= qa;
//               qb <= qb;
//               q  <= q;
//            end
//      endcase
//   else
//      begin
//         qa <= qa;
//         qb <= qb;
//         q  <= q;
//      end
//end
//
//endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Accumulator with parameterized width                      //
//                                                           //
///////////////////////////////////////////////////////////////

module accum(clk,clr,ce,in,out);
parameter in_width = 16; // Default input width in bits
parameter out_width = 24; // Default output width in bits

input clk;
input clr;
input ce;
input [in_width-1:0] in;
output [out_width-1:0] out;

reg [out_width-1:0] out;

always @ (posedge clk or posedge clr)
begin
   if (clr)
      out <= 0;
   else
      if (ce)
         out <= out + in;
      else
         out <= out;
end
endmodule

///////////////////////////////////////////////////////////////
//                                                           //
// Transmit Processor Finite State Machine                   //
// (for controling data on the transmit bus)                 //
//                                                           //
///////////////////////////////////////////////////////////////
module trans_proc(clk,rst,start,bg,mac_done,intr_err,fatal_err,s_err,data_done,bus_clr,hndshk,txp_st,br);

 input clk; 
 input rst; 
 input start; 
 input bg; 
 input mac_done; 
 input intr_err; 
 input fatal_err; 
 input s_err; 
 input data_done; 
 input bus_clr; 
 input hndshk;
output [3:0] txp_st;
output br;

reg [3:0] txp_st;
reg br;

always @(posedge clk or posedge rst)
begin: TxP_FSM
   if(rst)
      txp_st <= `TxP_Idle;
   else
      case(txp_st)
         `TxP_Idle:
            if(start)
               txp_st<= `TxP_BR;
            else
               txp_st <= `TxP_Idle;
         `TxP_BR:
            if(bg)
               txp_st<= `TxP_Ini_Stat;
            else
               txp_st <= `TxP_BR;
         `TxP_Ini_Stat:
            txp_st <= `TxP_MAC;
         `TxP_MAC:
            if(mac_done)
               txp_st<= `TxP_Data;
            else
               txp_st <= `TxP_MAC;
         `TxP_Data:
            if(data_done)
               txp_st<= `TxP_Fin_Stat;
            else if(fatal_err && !s_err)
               txp_st<= `TxP_Inc_Stat;
            else if(intr_err && !s_err)
               txp_st<= `TxP_Mid_Stat;
            else
               txp_st <= `TxP_Data;
         `TxP_Mid_Stat:
            txp_st <= `TxP_Clr_Intr;
         `TxP_Clr_Intr:
            if(!intr_err)
               txp_st <= `TxP_Data;
            else
               txp_st <= `TxP_Clr_Intr;
         `TxP_Inc_Stat:
            txp_st <= `TxP_Send_Pkt;
         `TxP_Fin_Stat:
            txp_st <= `TxP_Send_Pkt;
         `TxP_Send_Pkt:
            txp_st <= `TxP_Flush;
         `TxP_Flush:
            if(bus_clr)
               txp_st<= `TxP_Rel_Bus;
            else
               txp_st <= `TxP_Flush;
         `TxP_Rel_Bus:
            if(!bg)
               txp_st<= `TxP_Done;
            else
               txp_st <= `TxP_Rel_Bus;
         `TxP_Done:
            if(hndshk)
               txp_st<= `TxP_Idle;
            else
               txp_st <= `TxP_Done;
         default:
            txp_st <= `TxP_Idle;
      endcase
end
always @(txp_st)
begin
   case(txp_st)
      `TxP_BR,`TxP_Ini_Stat,`TxP_MAC,`TxP_Data,
      `TxP_Fin_Stat,`TxP_Send_Pkt,`TxP_Flush:
         br <= 1;
      default:
         br <= 0;
   endcase
end
endmodule
