///////////////////////////////////////////////////////////////
//                                                           //
// FIFO for ethernet frames                                  //
//                                                           //
///////////////////////////////////////////////////////////////

`timescale 1ns / 1ps
`ifndef HEADER
`include "header.v"
`endif


module ff_ctrl(CLK, RXRESET, FRAME_STAT_VALID, GOOD_FRAME, RXF_BUSY, READ,
      RXVALDAT, MAC_PUSH, WADDR, RADDR, FRAME_LEN, FRAME_READY, MT, FULL, AF,
      LEN_RST, LEN_LOAD, LEN_POP, LEN_PUSH, LOAD, SAVE, POP, PUSH, RESET, RP_RST, WP_RST);

    input CLK;
    input RXRESET;
    input FRAME_STAT_VALID;
    input GOOD_FRAME;
    input RXF_BUSY;
    input READ;
    input RXVALDAT;
    input MAC_PUSH;
    input [12:0] WADDR;
    input [12:0] RADDR;
    input [12:0] FRAME_LEN;
   output FRAME_READY;
   output MT;
   output FULL;
   output AF;
   output LEN_RST;
   output LEN_LOAD;
   output LEN_POP;
   output LEN_PUSH;
   output LOAD;
   output SAVE;
   output POP;
   output PUSH;
   output RESET;
   output RP_RST;
   output WP_RST;


reg [2:0] RX_WRT_ST;
reg [1:0] RX_RD_ST; 
reg RESET;
reg FRAME_READY;
reg WP_RST;
reg RP_RST;
reg LOAD;
reg SAVE;
reg LEN_RST, LEN_PUSH, LEN_POP, LEN_LOAD;
reg PK_DIR, PK_CE;
reg WRD_DIR, WRD_CE, WRD_DEC, WRD_RST_LD, WRD_CNT_LD;
reg [12:0] WRD_SAV, WRD_LOAD;


wire [12:0] WADDR;
wire [12:0] RADDR;
wire [12:0] FRAME_LEN;
wire [9:0] PKT_CNT;
wire [12:0] WRD_CNT;
wire [12:0] WRD_RST;
wire PUSH;
wire POP;
wire MT, FULL, AF, LEN_MT;


assign   PUSH = (RXVALDAT && !FULL);
assign    POP = (READ && !MT);
assign LEN_MT = (PKT_CNT == 10'd0);
assign     MT = (WRD_CNT == 13'd0);
assign     AF = (WRD_CNT >= `Rx_AF_Cnt);
assign   FULL = (WRD_CNT == `Rx_Full_Cnt);

     

always @ (posedge CLK or posedge RXRESET)
begin: RX_FIFO_Write_control
   if (RXRESET)
      RX_WRT_ST <= `RST_all;
   else
      case (RX_WRT_ST)
         `RST_all:
            RX_WRT_ST <= `Save_Addr; 
         `Save_Addr:
            RX_WRT_ST <= `Wait_for_Data; 
         `Wait_for_Data: 
            casex ({PUSH,MAC_PUSH})
               2'b00:
                  RX_WRT_ST <= `Wait_for_Data;
               2'b01: 
                  RX_WRT_ST <= `Store_MAC_Source;
               2'b1x: 
                  RX_WRT_ST <= `Store_Frame_Data;
               default:
                  RX_WRT_ST <= `Wait_for_Data;
            endcase
         `Store_MAC_Source:
            if (MAC_PUSH)
               RX_WRT_ST <= `Store_MAC_Source;
            else 
               RX_WRT_ST <= `Wait_for_Data;
         `Store_Frame_Data: 
            casex ({FRAME_STAT_VALID, GOOD_FRAME})
               2'b0x:
                  RX_WRT_ST <= `Store_Frame_Data;
               2'b10: 
                  RX_WRT_ST <= `Bad_Frame_St;
               2'b11: 
                  RX_WRT_ST <= `Good_Frame_St;
               default:
                  RX_WRT_ST <= `Bad_Frame_St;
            endcase
         `Bad_Frame_St: 
            RX_WRT_ST <= `Wait_for_Data; 
         `Good_Frame_St: 
            RX_WRT_ST <= `Save_Addr;
         default: 
            RX_WRT_ST <= `RST_all;
      endcase
end
always @(RX_WRT_ST)
begin 
   case (RX_WRT_ST) 
      `RST_all:
         begin
            WP_RST <= 1'b1;
            RP_RST <= 1'b1;
         end
      default:
         begin
            WP_RST <= 1'b0;
            RP_RST <= 1'b0;
         end
  endcase
end

    always @ (RX_WRT_ST)
    begin: RX_FIFO_Write_control_Outputs  
       case (RX_WRT_ST) 
          `RST_all:
             begin
                RESET <= 1'b1;
                LOAD        <= 1'b0;
                SAVE <= 1'b0;
                LEN_RST        <= 1'b1;
                LEN_PUSH <= 1'b0;
             end
          `Save_Addr:
             begin
                RESET <= 1'b0;
                LOAD        <= 1'b0;
                SAVE <= 1'b1;
                LEN_RST        <= 1'b0;
                LEN_PUSH <= 1'b0;
             end
          `Wait_for_Data:
             begin
                RESET <= 1'b0;
                LOAD        <= 1'b0;
                SAVE <= 1'b0;
                LEN_RST        <= 1'b0;
                LEN_PUSH <= 1'b0;
             end
          `Store_MAC_Source:
             begin
                RESET <= 1'b0;
                LOAD        <= 1'b0;
                SAVE <= 1'b0;
                LEN_RST        <= 1'b0;
                LEN_PUSH <= 1'b0;
             end
          `Store_Frame_Data:
             begin
                RESET <= 1'b0;
                LOAD        <= 1'b0;
                SAVE <= 1'b0;
                LEN_RST        <= 1'b0;
                LEN_PUSH <= 1'b0;
             end
          `Bad_Frame_St:
             begin
                RESET <= 1'b0;
                LOAD        <= 1'b1;
                SAVE <= 1'b0;
                LEN_RST        <= 1'b1;
                LEN_PUSH <= 1'b0;
             end
          `Good_Frame_St:
             begin
                RESET <= 1'b0;
                LOAD        <= 1'b0;
                SAVE <= 1'b0;
                LEN_RST        <= 1'b1;
                LEN_PUSH <= 1'b1;
             end
          default:
             begin
                RESET <= 1'b0;
                LOAD        <= 1'b0;
                SAVE <= 1'b0;
                LEN_RST        <= 1'b0;
                LEN_PUSH <= 1'b0;
             end
      endcase
    end
    
    always @ (posedge CLK or posedge RXRESET)
    begin: RX_FIFO_Read_control
       if (RXRESET)
          RX_RD_ST <= `Wait_for_Frame;
       else
          case (RX_RD_ST)
             `Wait_for_Frame:
                if (!LEN_MT && !RXF_BUSY)
                   RX_RD_ST <= `Load_Cnt_Dwn;
                else
                   RX_RD_ST <= `Wait_for_Frame;
             `Load_Cnt_Dwn:
                RX_RD_ST <= `Issue_Frm_Rdy; 
             `Issue_Frm_Rdy:
                if ((FRAME_LEN == 13'd0) || (MT))
                   RX_RD_ST <= `Wait_for_Frame;
                else
                   RX_RD_ST <= `Issue_Frm_Rdy;
             default: 
                RX_RD_ST <= `Wait_for_Frame;
          endcase
    end

    always @ (RX_RD_ST or LEN_MT or FRAME_LEN or MT or RXF_BUSY)
    begin: RX_FIFO_Read_control_Outputs  
       case (RX_RD_ST) 
          `Wait_for_Frame:
             begin
                LEN_POP <= !LEN_MT && !RXF_BUSY;
                LEN_LOAD <= 1'b0;
                FRAME_READY <= 1'b0;
             end
          `Load_Cnt_Dwn:
             begin
                LEN_POP <= 1'b0;
                LEN_LOAD <= 1'b1;
                FRAME_READY <= 1'b0;
             end
          `Issue_Frm_Rdy:
             begin
                LEN_POP <= 1'b0;
                LEN_LOAD <= 1'b0;
                FRAME_READY <= !((FRAME_LEN == 13'd0) || MT);
             end
          default:
             begin
                LEN_POP <= 1'b0;
                LEN_LOAD <= 1'b0;
                FRAME_READY <= !((FRAME_LEN == 13'd0) || MT);
             end
      endcase
    end
    
     
///////////////////////////////////////////////////////////////////////////////
// Packet counter for total number of packets currently waiting in the FIFO  //
///////////////////////////////////////////////////////////////////////////////

always @ (LEN_PUSH or LEN_POP or LEN_MT)
begin: Pkt_cntr_ctrl
  casex ({LEN_MT,LEN_PUSH,LEN_POP})
     3'b001:
       begin                  // count down
         PK_CE <= 1'b1;
         PK_DIR <= 1'b0;
       end
     3'bx10,3'b111:
       begin
         PK_CE <= 1'b1;          // count up
         PK_DIR <= 1'b1;
       end
     default:                  // hold
       begin
         PK_CE <= 1'b0;
         PK_DIR <= 1'b1;
       end
  endcase
end

defparam   pkt_cntr.width = 10;
cnt_up_dwn pkt_cntr (.clk(CLK),.ce(PK_CE),.dir(PK_DIR),.r(RESET),.count(PKT_CNT));

//////////////////////


/////////////////////////////////////////////////////
// Word counter  for total number of words in FIFO //
/////////////////////////////////////////////////////

always @ (PUSH or POP or MT or FULL or LOAD or WRD_RST)
begin: Wrd_Cnt_Ctrl
  casex ({FULL,PUSH,MT,POP,LOAD})
     5'bx0010,5'b11010:
       begin                     // count down
         WRD_CE <= 1'b1;
         WRD_DIR <= 1'b0;
         WRD_CNT_LD <= 1'b0;
         WRD_LOAD <= WRD_RST;
       end
     5'b01x00,5'b01110:
       begin                     // count up
         WRD_CE <= 1'b1;     
         WRD_DIR <= 1'b1;
         WRD_CNT_LD <= 1'b0;
         WRD_LOAD <= WRD_RST;
       end
     5'bX0001:
       begin                     // Load
         WRD_CE <= 1'b1;     
         WRD_DIR <= 1'b1;
         WRD_CNT_LD <= 1'b1;
         WRD_LOAD <= WRD_RST;
       end
     5'bX0011:
       begin                     // Load with decrement
         WRD_CE <= 1'b1;     
         WRD_DIR <= 1'b1;
         WRD_CNT_LD <= 1'b1;
         WRD_LOAD <= WRD_RST-1;
       end
     default:                    // hold
       begin
         WRD_CE <= 1'b0;
         WRD_DIR <= 1'b1;
         WRD_CNT_LD <= 1'b0;
         WRD_LOAD <= WRD_RST;
       end
  endcase
end

defparam     wrd_cntr.width = 13;
cnt_up_dwn_l wrd_cntr (.clk(CLK),.ce(WRD_CE),.dir(WRD_DIR),.r(RESET),.l(WRD_CNT_LD),.din(WRD_LOAD),.count(WRD_CNT));

//////////////////////


/////////////////////////////////////////////////////////////////////
// Counter for maintaining word count if current packet is dropped //
/////////////////////////////////////////////////////////////////////
                                              
always @ (SAVE or POP or WRD_CNT)
begin: Wrd_Cnt_Rst_Ctrl
  case ({SAVE,POP})
     2'b11:                       
       begin                             // Save Word count -1 to WRD_RST counter
         WRD_SAV <= WRD_CNT-1;
         WRD_RST_LD <= 1'b1;
         WRD_DEC  <= 1'b0;
       end
     2'b10:                       
       begin                       // Save Word count        to WRD_RST counter
         WRD_SAV <= WRD_CNT;
         WRD_RST_LD <= 1'b1;
         WRD_DEC  <= 1'b0;
       end
     2'b01:
       begin                       // Decrement WRD_RST
         WRD_SAV <= WRD_CNT;
         WRD_RST_LD <= 1'b0;
         WRD_DEC  <= 1'b1;
       end
     default:                       
       begin                       // hold
         WRD_SAV <= WRD_CNT;
         WRD_RST_LD <= 1'b0;
         WRD_DEC  <= 1'b0;
       end
  endcase
end

defparam wrd_cnt_rst.width = 13;
cnt_dn_l wrd_cnt_rst (.clk(CLK),.dec(WRD_DEC),.r(RESET),.l(WRD_RST_LD),.din(WRD_SAV),.count(WRD_RST));

endmodule

