// Verilog model  - Wed Oct 13 07:39:12 2004

`timescale 1ns / 1ps

module vme_arbiter(SYSCLK, USRCLK, ARB_ENA, ARB_TYP, BBSY, BR, SYSRESET,
                   SYS_CTRL_ENA, TIME_OUT, SW2, ACKN,
                   BCLR, BG_ERR, BGIN, TEST_A, TEST_B, TEST_C);

    input SYSCLK;
    input USRCLK;
    input ARB_ENA;
    input [1:0] ARB_TYP;
    input BBSY;
    input [3:0] BR;
    input SYSRESET;
    input SYS_CTRL_ENA;
    input [15:0] TIME_OUT;
    input [3:0] SW2;
    input [3:0] ACKN;
   output BCLR;
   output BG_ERR;
   output [3:0] BGIN;
   output [15:0] TEST_A;
   output [15:0] TEST_B;
   output [15:0] TEST_C;
   
reg  ENA;
wire LEAD_EDG_BBSY;
wire TRAIL_EDG_BBSY;
wire BG_TM_OUT;
wire TMR_RST;

reg DB0,DB1,DB2,DB3,DB4,DB5;   

reg [1:0] SGL_ST;
reg [2:0] RRS_ST;
reg [2:0] PRI_ST;
reg [3:0] CURR_BUS;
reg [3:0] BGIN;
reg [15:0] BG_TMR;
reg BG_ERR;
reg ACKN_ERR;
reg BCLR;
reg [15:0] TEST_A;
reg [15:0] TEST_B;
reg [15:0] TEST_C;
   
`ifdef HEADER
`else
`include "header.v"
`endif

always @(posedge SYSCLK)
begin
   case (SW2)
      4'h5:
         begin
            TEST_A <= 16'h0000;
            TEST_B[3:0] <= BR;
            TEST_B[7:4] <= CURR_BUS;
            TEST_B[11:8] <= BGIN;
            TEST_B[12] <= ARB_ENA;
            TEST_B[13] <= BCLR;
            TEST_B[14] <= BG_ERR;
            TEST_B[15] <= BG_TM_OUT;
            TEST_C[1:0] <= ARB_TYP;
            TEST_C[4:2] <= PRI_ST;
            TEST_C[6:5] <= RRS_ST;
            TEST_C[7] <= BBSY;
            TEST_C[15:8] <= 8'h00;
         end
      default:
         begin
            TEST_A <= 16'h0000;
            TEST_B <= 16'h0000;
            TEST_C <= 16'h0000;
         end
   endcase
end


assign LEAD_EDG_BBSY = BBSY & !DB2 & !DB1 & !DB0;//eliminate false leading edges at trailing edge
assign TRAIL_EDG_BBSY = !BBSY & DB2 & DB1 & DB0; //eliminate false trailing edges at leading edge
assign BG_TM_OUT = (BG_TMR == TIME_OUT);
assign TMR_RST = SYSRESET || BG_ERR;

//BBSY delay for creating the disable region.
always @(posedge SYSCLK)
begin
   DB0 <= BBSY;
   DB1 <= DB0;   
   DB2 <= DB1;   
   DB3 <= DB2;   
   DB4 <= DB3;   
   DB5 <= DB4;   
   ENA <= ~|{DB5,DB4,DB3,DB2,DB1}; //40ns min. before new bus grant after BBSY* high
end

// Bus Grant Timer
always @(posedge USRCLK or posedge TMR_RST)
begin
   if(TMR_RST)
      BG_TMR <= 16'h0000;
   else
      if(|BGIN && !BBSY)
         BG_TMR <= BG_TMR + 1;
      else if(BBSY)
         BG_TMR <= 16'h0000;
      else
         BG_TMR <= BG_TMR;   
end

always @(posedge SYSCLK)
begin
   ACKN_ERR <= (ACKN == `V_Arb_Err);
end

always @(posedge SYSCLK or posedge SYSRESET)
begin
   if(SYSRESET)
      BG_ERR <= 0;
   else
      if(BG_TM_OUT && !BBSY)
         BG_ERR <= 1;
      else if(ACKN_ERR)
         BG_ERR <= 0;
      else
         BG_ERR <= BG_ERR;   
end 



//State machine for Single Level Arbitration.
always @(posedge SYSCLK or posedge SYSRESET)
begin: VME_SGL_ARB_FSM
   if(SYSRESET)
      SGL_ST <= `Mon_BR;
   else 
      case(SGL_ST)
         `Mon_BR3:
            begin
               if(BR[3] && !BBSY && ENA)
                  SGL_ST <= `BG3;
               else
                  SGL_ST <= `Mon_BR3;
            end
         `BG3:
            begin
               if(BBSY)
                  SGL_ST <= `Mon_BR3;
               else if(BG_ERR)
                  SGL_ST <= `BG_Err_S;
               else
                  SGL_ST <= `BG3;
            end
         `BG_Err_S:
            begin
               if(ACKN_ERR)
                  SGL_ST <= `Mon_BR3;
               else
                  SGL_ST <= `BG_Err_S;
            end
         default:
            SGL_ST <= `Mon_BR3;
      endcase
end

//State machine for Prioritized Arbitration.
always @(posedge SYSCLK or posedge SYSRESET)
begin: VME_PRIO_ARB_FSM
   if(SYSRESET)
      PRI_ST <= `Mon_BR;
   else 
      case(PRI_ST)
         `Mon_BR:
            begin
               if(!BBSY && ENA)
                  casex (BR)
                     4'b1xxx:
                        PRI_ST <= `BG_lvl3;
                     4'b01xx:
                        PRI_ST <= `BG_lvl2;
                     4'b001x:
                        PRI_ST <= `BG_lvl1;
                     4'b0001:
                        PRI_ST <= `BG_lvl0;
                     default:
                        PRI_ST <= `Mon_BR;
                  endcase
               else
                  PRI_ST <= `Mon_BR;
            end
         `BG_lvl3:
            begin
               if(BBSY)
                  PRI_ST <= `Mon_BR;
               else if(BG_ERR)
                  PRI_ST <= `BG_Err_P;
               else
                  PRI_ST <= `BG_lvl3;
            end
         `BG_lvl2:
            begin
               if(BBSY)
                  PRI_ST <= `Mon_BR;
               else if(BG_ERR)
                  PRI_ST <= `BG_Err_P;
               else
                  PRI_ST <= `BG_lvl2;
            end
         `BG_lvl1:
            begin
               if(BBSY)
                  PRI_ST <= `Mon_BR;
               else if(BG_ERR)
                  PRI_ST <= `BG_Err_P;
               else
                  PRI_ST <= `BG_lvl1;
            end
         `BG_lvl0:
            begin
               if(BBSY)
                  PRI_ST <= `Mon_BR;
               else if(BG_ERR)
                  PRI_ST <= `BG_Err_P;
               else
                  PRI_ST <= `BG_lvl0;
            end
         `BG_Err_P:
            begin
               if(ACKN_ERR)
                  PRI_ST <= `Mon_BR;
               else
                  PRI_ST <= `BG_Err_P;
            end
         default:
            PRI_ST <= `Mon_BR;
      endcase
end

//State machine for Round Robin Select Arbitration.
always @(posedge SYSCLK or posedge SYSRESET)
begin: VME_RROB_ARB_FSM
   if(SYSRESET)
      RRS_ST <= `RRS3;
   else 
      case(RRS_ST)
         `RRS3:
            begin
               if(!BR[3] && !BBSY)
                  RRS_ST <= `RRS2;
               else if(BG_ERR)
                  RRS_ST <= `BG_Err_RR3;
               else
                  RRS_ST <= `RRS3;
            end
         `RRS2:
            begin
               if(!BR[2] && !BBSY)
                  RRS_ST <= `RRS1;
               else if(BG_ERR)
                  RRS_ST <= `BG_Err_RR2;
               else
                  RRS_ST <= `RRS2;
            end
         `RRS1:
            begin
               if(!BR[1] && !BBSY)
                  RRS_ST <= `RRS0;
               else if(BG_ERR)
                  RRS_ST <= `BG_Err_RR1;
               else
                  RRS_ST <= `RRS1;
            end
         `RRS0:
            begin
               if(!BR[0] && !BBSY)
                  RRS_ST <= `RRS3;
               else if(BG_ERR)
                  RRS_ST <= `BG_Err_RR0;
               else
                  RRS_ST <= `RRS0;
            end
         `BG_Err_RR3:
            begin
               if(ACKN_ERR)
                  RRS_ST <= `RRS2;
               else
                  RRS_ST <= `BG_Err_RR3;
            end
         `BG_Err_RR2:
            begin
               if(ACKN_ERR)
                  RRS_ST <= `RRS1;
               else
                  RRS_ST <= `BG_Err_RR2;
            end
         `BG_Err_RR1:
            begin
               if(ACKN_ERR)
                  RRS_ST <= `RRS0;
               else
                  RRS_ST <= `BG_Err_RR1;
            end
         `BG_Err_RR0:
            begin
               if(ACKN_ERR)
                  RRS_ST <= `RRS3;
               else
                  RRS_ST <= `BG_Err_RR0;
            end
         default:
            RRS_ST <= `RRS3;
      endcase
end


// Logic for BG(3:0)IN
always @(posedge SYSCLK or posedge SYSRESET)
begin
   if(SYSRESET)
      BGIN <= 4'h0;
   else 
      if(ARB_ENA && SYS_CTRL_ENA)
         case (ARB_TYP)
            `No_Arb:        // disabled
               BGIN <= 4'h0;
            `SGL:        // Single Level Arbiter (SGL)
               case (SGL_ST)
                  `Mon_BR3:
                     BGIN <= 4'h0;
                  `BG3:
                     BGIN <= 4'h8;
                  `BG_Err_S:
                     BGIN <= 4'h0;
                  default:
                     BGIN <= 4'h0;
               endcase
            `PRI:        // Prioritized Arbiter (PRI)
               case (PRI_ST)
                  `Mon_BR:
                     BGIN <= 4'h0;
                  `BG_lvl3:
                     begin
                        BGIN[3] <= 1;
                        BGIN[2] <= 0;
                        BGIN[1] <= 0;
                        BGIN[0] <= 0;
                     end
                  `BG_lvl2:
                     begin
                        BGIN[3] <= 0;
                        BGIN[2] <= 1;
                        BGIN[1] <= 0;
                        BGIN[0] <= 0;
                     end
                  `BG_lvl1:
                     begin
                        BGIN[3] <= 0;
                        BGIN[2] <= 0;
                        BGIN[1] <= 1;
                        BGIN[0] <= 0;
                     end
                  `BG_lvl0:
                     begin
                        BGIN[3] <= 0;
                        BGIN[2] <= 0;
                        BGIN[1] <= 0;
                        BGIN[0] <= 1;
                     end
                  `BG_Err_P:
                     BGIN <= 4'h0;
                  default:
                     BGIN <= 4'h0;
               endcase
            `RRS:        // Round-Robin-Select Arbiter (RRS)
               case (RRS_ST)
                  `RRS3:
                     begin
                        BGIN[3] <= BR[3] & !BBSY & ENA;
                        BGIN[2] <= 0;
                        BGIN[1] <= 0;
                        BGIN[0] <= 0;
                     end
                  `RRS2:
                     begin
                        BGIN[3] <= 0;
                        BGIN[2] <= BR[2] & !BBSY & ENA;
                        BGIN[1] <= 0;
                        BGIN[0] <= 0;
                     end
                  `RRS1:
                     begin
                        BGIN[3] <= 0;
                        BGIN[2] <= 0;
                        BGIN[1] <= BR[1] & !BBSY & ENA;
                        BGIN[0] <= 0;
                     end
                  `RRS0:
                     begin
                        BGIN[3] <= 0;
                        BGIN[2] <= 0;
                        BGIN[1] <= 0;
                        BGIN[0] <= BR[0] & !BBSY & ENA;
                     end
                  default:
                     BGIN <= 4'h0;
               endcase
            default:
               BGIN <= 4'h0;
         endcase
      else
         BGIN <= 4'h0;
end



// Hold current Bus Grant line while the bus is busy.  
always @(posedge SYSCLK or posedge SYSRESET)
begin
   if(SYSRESET)
      CURR_BUS <= 4'h0;
   else
      if(LEAD_EDG_BBSY)
         CURR_BUS <= BGIN;
      else if(TRAIL_EDG_BBSY)
         CURR_BUS <= 4'h0;
      else
         CURR_BUS <= CURR_BUS;
end

// Logic for BCLR

always @(posedge SYSCLK or posedge SYSRESET)
begin
   if(SYSRESET)
      BCLR <= 0;
   else 
      if(ARB_ENA && SYS_CTRL_ENA)
         begin
            case (ARB_TYP)
               `PRI:        // Prioritized Arbiter (PRI)
                  begin
                     if(BBSY && DB1)
                        if(BR > CURR_BUS)
                           BCLR <= 1;
                        else
                           BCLR <= 0;
                     else
                        BCLR <= 0;
                  end
               `RRS:        // Round-Robin-Select Arbiter (RRS)
                  begin
                     if(CURR_BUS == 4'h0)
                        BCLR <= 0;
                     else
                        BCLR <= |(BR&~CURR_BUS);
                  end
               default:
                  BCLR <= 0;
            endcase
         end
      else
         BCLR <= 0;
end
   
endmodule
